%%  Functional Flows
%%
%%  Written by Stephen Murray and  Marisa Escobar
%%  April 25, 2007
%%  UC Davis Land, Air, and Water Resources

clc
clear
close all
format compact
format short g
figure_counter = 1;

disp('**************************************************')
disp('*************    Functional Flows    *************')
disp('**************************************************')
disp(' ')
disp(' ')

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%  Units  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Ask the user for English or Metric units.

units = 0;
while units == 0
    
    disp('<<<<   Units System   >>>>')
    disp(' ')
    disp('In what units are your data entries?')
    disp('1. I am using Metric units.')
    disp('2. I am using English units.')
    disp(' ')
    
    units = input('Please enter 1 or 2:  ');
    
    %Make sure the user only enters 1 or 2.
    if units ~= 1 && units ~= 2
        disp(' ')
        disp('Please choice either 1 or 2.')
        disp(' ')
        units = 0;
    end
    
end
% The value of units will be 1 for metric, 2 for english
if units == 1
    ro = 1000;  %kg/m^3
    ros = 2600; %kg/m^3
    g = 9.8;   %m/s^2
else
    ro = 1.9403;    %slug/ft^3
    ros = 5.0448;   %slugs/ft^3
    g = 32.15;      %ft/s    
end

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%  Slope  %%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get the slope from the get_slope function.

[Hi_slope Lo_slope] = get_slope;

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%  Obtaining c and f  %%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get the data from the Manning's equation.

[c_r,f_r,new_geo,S,inc,figure_counter,c_r_H,f_r_H,...
    w_r_Man,A_r_Man,P_r_Man,R_r_Man,V_r_Man,Q_r_Man,H_r_Man,stage_r] =...
    get_mannings_data(Hi_slope,Lo_slope, units, figure_counter);

%% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%  D50  %%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get D50, the average grain size.

[D50,D50_choice,n_gs,gs_range] = get_D50(units);

%% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%   Flow Data   %%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Getting the flow data
% This explanation may be saved for the user's manual
disp(' ')
disp('<<<<   Daily Flow Data   >>>>')
disp(' ')
disp('Please copy your flow data into the folder in which this program runs.')
disp('The flow data should contain four columns.')
disp('The first 3 columns are month, day, and year respectively, beginning with Oct 1.')
disp('The last column is the flow rate, Q.')
disp('The analysis of multiple water years is possible.')
disp('Simply add the same data for a new year just below the previous year''s data.')
disp('The program will automatically detect the number of years.')
disp('Please ensure the years are in chronologial order.')
disp(' ')

if units == 1
    disp('Since you are in Metric units, make sure the value of Q is in m^3/s.')
    disp(' ')
else
    disp('Since you are in English units, make sure the value of Q is in ft^3/s.')
    disp(' ')
end

flow_string = input('Please enter the name of the flow data file, including the .txt extension:  ','s');
flow_data = load(flow_string);
Q = flow_data(:,4);
julian_day = 1:1:length(Q);

%Find the number of years in the flow data
years_vector = flow_data(:,3);

keep_going = 1;
test_year = years_vector(1);
i = 1;
while keep_going
    year_holder = find(years_vector > test_year);    
    
    if isempty(year_holder)
        keep_going = 0;
    else
        water_years(i) = years_vector(year_holder(1));
    
        test_year = water_years(i);
        i = i + 1;
        keep_going = 1;
    end %if
        
end %while

n_water_years = length(water_years);

for i = 1:n_water_years
    
    % General case
    if mod(water_years(i),4) == 0        
        is_leap_year(i) = 1;        
    else
        is_leap_year(i) = 0;
    end %if
    
    % Case of centurial years: only centurial years divisible by 400 are
    % leap years
    if mod(water_years(i),100) == 0
        if mod(water_years(i),400) == 0
            is_leap_year(i) = 1;
        else
            is_leap_year(i) = 0;
        end %inner if
    end %outer if
    
end %i loop

% Get a vector to test the julian day
julian_test = [];

if n_water_years > 1
    
    for k = 1:n_water_years - 1
        p = length(find(years_vector == water_years(k)));
        julian_test = [julian_test,1:1:p];
    end
    
    k = n_water_years;
    julian_test = [julian_test,1:1:(365 + is_leap_year(k))];
else
    julian_test = 1:1:(365 + is_leap_year);
end

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%  Extra Tau Terms  %%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get any extra terms in the equation for tau.

[dhdx, nonuniform, unsteady, new_geo_p, figure_counter, new_geo, c_p, f_p,...
    c_p_H,f_p_H,w_p_Man,A_p_Man,P_p_Man,R_p_Man,V_p_Man,Q_p_Man,H_p_Man,stage_p,Hi_slope_p,Lo_slope_p...
    c_r_H,f_r_H,w_r_Man,A_r_Man,P_r_Man,R_r_Man,V_r_Man,Q_r_Man,H_r_Man,stage_r] =...
    get_extra_tau_terms(Q,units,g,c_r,f_r,new_geo,figure_counter,c_r_H,f_r_H,Hi_slope,Lo_slope,...
    w_r_Man,A_r_Man,P_r_Man,R_r_Man,V_r_Man,Q_r_Man,H_r_Man,stage_r);

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%   Functionality Table  %%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get the functionality data.

[tau_thresh, matrix, n_eco_func, start_day, end_day, ... 
    n_ranges, firstCol, eco_names, spaces, n_chunks, eco_func_per_chunk, ...
    final_rows, func_choice] = get_func(julian_day);

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%   Visualizing Inputs  %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Call the function to ask if the user wants to see the inputs
[figure_counter] = visuals(new_geo, Q, units, n_ranges, n_eco_func,...
    matrix, firstCol, start_day, end_day, eco_names, spaces, julian_day,...
    figure_counter, n_chunks, eco_func_per_chunk, final_rows, func_choice, ...
    new_geo_p);

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%   Checking Inputs   %%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[dhdx, nonuniform, unsteady,S,figure_counter,Hi_slope,Lo_slope,new_geo,new_geo_p,c_r,f_r,...
    func_choice,n_chunks,final_rows,n_ranges,firstCol,eco_func_per_chunk,...
    matrix,spaces,julian_day,Q,eco_names,n_eco_func,start_day,end_day,...
    c_r_H,f_r_H,w_r_Man,A_r_Man,P_r_Man,R_r_Man,V_r_Man,Q_r_Man,H_r_Man,stage_r,D50,D50_choice,n_gs,gs_range,...
    c_p, f_p,c_p_H,f_p_H,w_p_Man,A_p_Man,P_p_Man,R_p_Man,V_p_Man,Q_p_Man,H_p_Man,stage_p,Hi_slope_p,Lo_slope_p,tau_thresh] =...
    check_inputs(units,g,figure_counter,Hi_slope,Lo_slope,new_geo,new_geo_p,...
    c_r,f_r,func_choice,n_chunks,final_rows,n_ranges,firstCol,...
    eco_func_per_chunk,matrix,spaces,julian_day,Q,eco_names,n_eco_func,...
    start_day,end_day,c_r_H,f_r_H,w_r_Man,A_r_Man,P_r_Man,R_r_Man,...
    V_r_Man,Q_r_Man,H_r_Man,stage_r,dhdx, nonuniform, unsteady,S,D50,D50_choice,n_gs,gs_range,...
    c_p, f_p,c_p_H,f_p_H,w_p_Man,A_p_Man,P_p_Man,R_p_Man,V_p_Man,Q_p_Man,H_p_Man,stage_p,Hi_slope_p,Lo_slope_p,tau_thresh);



%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%   Computing the Shear  %%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This section will compute the ranges where the slope increments will be
% used and finds the quantity (cQ^f)*S in the equation for the
% non-dimensionalized shear stress
Qmax = max(Q);
Qmin = min(Q);

dQtest = ((Qmax-Qmin)/(length(S)-1));

for j=1:length(S)
    Qtest(j) = Qmin + dQtest*(j-1);
    Qtest(j) = Roundoff(Qtest(j),10);
end

% This double loop calculates the shear stress and flow
for i=1:length(Q)    
    
    for j = 1:length(S) 
        
        if (Q(i) >= Qtest(j)) && (Q(i) < (Qtest(j)+dQtest))
            tau(i) = ro*g*(c_r*Q(i)^f_r)*(S(j) - dhdx(i) - nonuniform(i) - unsteady(i));
            tau_star(i) = tau(i)/(g*(ros - ro)*D50);            
            break
        end  % if
        
    end % j loop
    
    Q_star(i) = Q(i)/(sqrt(g*D50)*D50^2);
    
end % i loop

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%   Displaying the Outputs  %%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This section finally displays the outputs
disp(' ')
disp(' ')
disp('** The program will now calculate the outputs. **')
[figure_counter,GMF,real_F,total_days_GMF,total_days_F,t_store,Q_store,tau_is_pos] ...
    = outputs(tau_thresh, n_ranges, n_eco_func, eco_names, tau_star, Q_star,...
    matrix, start_day, end_day, julian_day, figure_counter, julian_test,...
    n_chunks, eco_func_per_chunk, units);

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%   Writing the Outputs  %%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

writing_outputs(units,Hi_slope,Lo_slope,new_geo,new_geo_p,...
    c_r,f_r,c_p,f_p,D50,func_choice,n_chunks,final_rows,n_ranges,...
    firstCol,eco_func_per_chunk,matrix,spaces,julian_day,Q,tau_star,...
    Q_star,eco_names,GMF,real_F,n_eco_func,total_days_GMF,total_days_F,...
    start_day,end_day,D50_choice,n_gs,gs_range,...
    c_r_H,f_r_H,w_r_Man,A_r_Man,P_r_Man,R_r_Man,V_r_Man,Q_r_Man,H_r_Man,stage_r,...
    c_p_H,f_p_H,w_p_Man,A_p_Man,P_p_Man,R_p_Man,V_p_Man,Q_p_Man,H_p_Man,stage_p,Hi_slope_p,Lo_slope_p,...
    t_store,Q_store,tau_thresh,tau_is_pos)

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%    Sensitivity Study    %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sensitivity(units,ro,ros,g,figure_counter,Hi_slope,Lo_slope,new_geo,new_geo_p,...
    c_r,f_r,c_p,f_p,D50,func_choice,n_chunks,final_rows,n_ranges,...
    firstCol,eco_func_per_chunk,matrix,spaces,julian_day,Q,tau_star,...
    Q_star,eco_names,n_eco_func,start_day,end_day,D50_choice,n_gs,gs_range,...
    c_r_H,f_r_H,w_r_Man,A_r_Man,P_r_Man,R_r_Man,V_r_Man,Q_r_Man,H_r_Man,stage_r,...
    c_p_H,f_p_H,w_p_Man,A_p_Man,P_p_Man,R_p_Man,V_p_Man,Q_p_Man,H_p_Man,stage_p,Hi_slope_p,Lo_slope_p,...
    tau_thresh,S,dhdx,nonuniform,unsteady,julian_test)